<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log; // Optional: for logging errors
use App\Http\Requests\CalculateLinkBudgetRequest;
use Exception; // Import Exception class

class LinkCalculatorController extends Controller
{
    //
    private const EARTH_RADIUS_KM = 6371.0;
    private const NUM_PROFILE_POINTS = 50; // Number of points for elevation profile

    /**
     * Calculate the link budget and elevation profile.
     *
     * @param CalculateLinkBudgetRequest $request
     * @return JsonResponse
     */
    public function calculate(CalculateLinkBudgetRequest $request): JsonResponse
    {
        $data = $request->validated();

        try {
            // 1. Calculate distance
            $distanceKm = $this->haversineDistance(
                $data['start_lat'], $data['start_lon'],
                $data['end_lat'], $data['end_lon']
            );

            if ($distanceKm <= 0) {
                return response()->json(['detail' => 'Start and end points are the same or invalid.'], 400);
            }

            // 2. Get elevation profile points
            $intermediateCoords = $this->calculateIntermediatePoints(
                $data['start_lat'], $data['start_lon'],
                $data['end_lat'], $data['end_lon'],
                self::NUM_PROFILE_POINTS
            );

            $elevations = $this->getElevations($intermediateCoords);

            if (empty($elevations)) {
                 return response()->json(['detail' => 'Failed to fetch elevation data.'], 500);
            }

            // 3. Calculate Fresnel Zones and LOS
            $wavelengthKm = (300 / ($data['frequency'] * 1000)) / 1000; // Wavelength in km
            $profilePoints = [];
            $hasLos = true; // Assume LOS initially

            $startElevation = $elevations[0] ?? 0;
            $endElevation = end($elevations) ?: 0;
            $startTotalHeight = $startElevation + $data['start_height'];
            $endTotalHeight = $endElevation + $data['end_height'];

            for ($i = 0; $i < count($elevations); $i++) {
                $pointDistance = $distanceKm * ($i / (self::NUM_PROFILE_POINTS - 1));
                $currentElevation = $elevations[$i];
                $currentCoords = $intermediateCoords[$i];

                // Linear interpolation for LOS path height
                $losHeight = $startTotalHeight + (($endTotalHeight - $startTotalHeight) * ($pointDistance / $distanceKm));

                // Fresnel zone radius (F1 = full radius, in meters)
                // Formula: R = 17.32 * sqrt( (d1 * d2) / (f * D) ) where d1, d2, D in km, f in GHz
                $d1 = $pointDistance;
                $d2 = $distanceKm - $pointDistance;
                $fresnelRadiusMeters = 0;
                if ($distanceKm > 0 && $data['frequency'] > 0 && $d1 > 0 && $d2 > 0) {
                    $fresnelRadiusMeters = 17.32 * sqrt(($d1 * $d2) / ($data['frequency'] * $distanceKm));
                }

                $f1Radius60Percent = $fresnelRadiusMeters * 0.6; // 60% of F1 radius
                $f2RadiusFull = $fresnelRadiusMeters;            // Full F1 radius

                $profilePoints[] = [
                    "lat" => $currentCoords['lat'],
                    "lng" => $currentCoords['lon'],
                    "distance" => $pointDistance,
                    "elevation" => $currentElevation,
                    "f1_radius" => $f1Radius60Percent, // 60% Fresnel zone radius
                    "f2_radius" => $f2RadiusFull,      // Full Fresnel zone radius
                    "los_height" => $losHeight,
                ];

                // Check for obstruction against 60% Fresnel zone
                $clearanceNeeded = $losHeight - $f1Radius60Percent;
                if ($currentElevation > $clearanceNeeded) {
                    $hasLos = false;
                    // No need to check further points if one obstruction found,
                    // but we continue loop to build the full profile data
                }
            }

            // 4. Calculate Link Budget
            // FSPL (dB) = 20*log10(d) + 20*log10(f) + 92.45 (d in km, f in GHz)
            $fspl = 20 * log10($distanceKm) + 20 * log10($data['frequency']) + 92.45;
            $receivedPower = $data['tx_power'] + $data['tx_antenna_gain'] + $data['rx_antenna_gain'] - $fspl;

            return response()->json([
                "distance_km" => $distanceKm,
                "fspl_dB" => $fspl,
                "received_power_dBm" => $receivedPower,
                "start_elevation" => $startElevation,
                "end_elevation" => $endElevation,
                "profile_points" => $profilePoints,
                "has_los" => $hasLos,
                "wavelength" => $wavelengthKm * 1000, // Return wavelength in meters
                "start_total_height" => $startTotalHeight,
                "end_total_height" => $endTotalHeight
            ]);

        } catch (Exception $e) {
            Log::error("Link Calculation Error: " . $e->getMessage(), ['request_data' => $data]); // Log error
            // Return a generic error message to the user
            return response()->json(['detail' => 'An error occurred during calculation. ' . $e->getMessage()], 500);
        }
    }

    /**
     * Calculate Haversine distance between two points.
     *
     * @param float $lat1
     * @param float $lon1
     * @param float $lat2
     * @param float $lon2
     * @return float Distance in kilometers
     */
    private function haversineDistance(float $lat1, float $lon1, float $lat2, float $lon2): float
    {
        $lat1Rad = deg2rad($lat1);
        $lon1Rad = deg2rad($lon1);
        $lat2Rad = deg2rad($lat2);
        $lon2Rad = deg2rad($lon2);

        $dLat = $lat2Rad - $lat1Rad;
        $dLon = $lon2Rad - $lon1Rad;

        $a = sin($dLat / 2) ** 2 + cos($lat1Rad) * cos($lat2Rad) * sin($dLon / 2) ** 2;
        $c = 2 * asin(sqrt($a));

        return self::EARTH_RADIUS_KM * $c;
    }

    /**
     * Generate intermediate points between start and end coordinates.
     *
     * @param float $startLat
     * @param float $startLon
     * @param float $endLat
     * @param float $endLon
     * @param int $numPoints
     * @return array Array of ['lat' => float, 'lon' => float]
     */
    private function calculateIntermediatePoints(float $startLat, float $startLon, float $endLat, float $endLon, int $numPoints): array
    {
        $points = [];
        if ($numPoints < 2) {
            return [['lat' => $startLat, 'lon' => $startLon]];
        }
        for ($i = 0; $i < $numPoints; $i++) {
            $fraction = $i / ($numPoints - 1);
            $points[] = [
                'lat' => $startLat + ($endLat - $startLat) * $fraction,
                'lon' => $startLon + ($endLon - $startLon) * $fraction
            ];
        }
        return $points;
    }

    /**
     * Get elevations for multiple points using Google Elevation API.
     *
     * @param array $coords Array of ['lat' => float, 'lon' => float]
     * @return array|null Array of elevation values (float) or null on error
     */
    private function getElevations(array $coords): ?array
    {
        $apiKey = env('GOOGLE_MAPS_API_KEY');
        if (empty($apiKey)) {
             Log::error("Google Maps API Key not configured.");
             throw new Exception("Server configuration error: Missing API Key.");
        }

        // Format locations for API call
        $locationsParam = implode('|', array_map(function ($coord) {
            return $coord['lat'] . ',' . $coord['lon'];
        }, $coords));

        $url = "https://maps.googleapis.com/maps/api/elevation/json";

        try {
            $response = Http::get($url, [
                'locations' => $locationsParam,
                'key' => $apiKey
            ]);

            if (!$response->successful()) {
                Log::error("Google Elevation API request failed.", [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                throw new Exception("Elevation API request failed with status: " . $response->status());
            }

            $data = $response->json();

            if ($data['status'] !== 'OK') {
                Log::error("Google Elevation API error.", [
                    'status' => $data['status'],
                    'error_message' => $data['error_message'] ?? 'No error message provided.'
                ]);
                 throw new Exception("Elevation API error: " . ($data['error_message'] ?? $data['status']));
            }

            $elevations = [];
            foreach ($data['results'] as $result) {
                $elevations[] = (float)$result['elevation'];
            }
            return $elevations;

        } catch (Exception $e) {
             Log::error("Exception during elevation fetch: " . $e->getMessage());
             // Re-throw specific exceptions or a generic one
             throw new Exception("Failed to retrieve elevation data due to a network or API issue.");
        }
}
